#include "UnityNP.h"
#include "MessagePipe.h"

#define CASE_ERROR_TO_STRING(errCode) case (int)errCode: errorString = #errCode; break;

namespace UnityPlugin
{
	bool gLogErrors = false;

	DO_EXPORT( bool, PrxGetLoggingEnabled ) ()
	{
		return gLogErrors;
	}

	DO_EXPORT( void, PrxSetLoggingEnabled ) (bool logErrors)
	{
		gLogErrors = logErrors;
	}

	ErrorCode ProcessSceResult(int sceError, bool log, const char* className, const char* function, int line, const char* message)
	{
		log &= gLogErrors;
		if(log && sceError != SCE_OK)
		{
			if(message)
			{
				UnityPlugin::Messages::LogError("%s::%s@L%d - %s - %s", className, function, line, message, LookupSceErrorCode(sceError));
			}
			else
			{
				UnityPlugin::Messages::LogError("%s::%s@L%d - %s", className, function, line, LookupSceErrorCode(sceError));
			}
		}
		return TranslateSceError(sceError);
	}

	ErrorCode ProcessSceResult(int sceError, bool log, const char* message)
	{
		log &= gLogErrors;
		if(log && sceError != SCE_OK)
		{
			UnityPlugin::Messages::LogError(message);
		}
		return TranslateSceError(sceError);
	}

	ErrorCode ProcessSceResult(int sceError, bool log)
	{
		log &= gLogErrors;
		if(log && sceError != SCE_OK)
		{
			UnityPlugin::Messages::LogError(LookupSceErrorCode(sceError));
		}
		return TranslateSceError(sceError);
	}

	ErrorCode ProcessResult(ErrorCode error, bool log, const char* className, const char* function, int line, const char* message)
	{
		log &= gLogErrors;
		if(log && !(error == NP_OK || error == NP_SIGNED_IN_FLIGHT_MODE))
		{
			if(message)
			{
				UnityPlugin::Messages::LogError("%s::%s@L%d - %s - %s", className, function, line, message, LookupErrorCode(error));
			}
			else
			{
				UnityPlugin::Messages::LogError("%s::%s@L%d - %s", className, function, line, LookupErrorCode(error));
			}
		}
		return error;
	}

	ErrorCode ProcessResult(ErrorCode error, bool log, const char* message)
	{
		log &= gLogErrors;
		if(log && !(error == NP_OK || error == NP_SIGNED_IN_FLIGHT_MODE))
		{
			UnityPlugin::Messages::LogError(message);
		}
		return error;
	}
	
	ErrorCode ProcessResult(ErrorCode error, bool log)
	{
		log &= gLogErrors;
		if(log && !(error == NP_OK || error == NP_SIGNED_IN_FLIGHT_MODE))
		{
			UnityPlugin::Messages::LogError(LookupErrorCode(error));
		}
		return error;
	}

	const char* LookupErrorCode(ErrorCode errorCode)
	{
		static char defaultErrorString[16];
		const char* errorString = NULL;

		switch(errorCode)
		{
			CASE_ERROR_TO_STRING(NP_OK)
			CASE_ERROR_TO_STRING(NP_ERR_FAILED)
			CASE_ERROR_TO_STRING(NP_ERR_BAD_STATE)
			CASE_ERROR_TO_STRING(NP_ERR_NOT_SIGNED_IN)
			CASE_ERROR_TO_STRING(NP_ERR_NOT_CONNECTED)
			CASE_ERROR_TO_STRING(NP_ERR_NOT_CONNECTED_FLIGHT_MODE)

		default:
			sprintf(defaultErrorString, "0x%08x", errorCode);
			errorString = defaultErrorString;
			break;
		}

		return errorString;
	}

	ResultCode::ResultCode(const char* className)
		: m_className(className)
		, m_LastError(NP_OK)
		, m_LastErrorSCE(0)
	{
	}

	ErrorCode ResultCode::SetResultSCE(int sceError, bool log, const char* function, int line, const char* message)
	{
		m_LastErrorSCE = sceError;
		m_LastError = ProcessSceResult(sceError, log, m_className, function, line, message);
		return m_LastError;
	}

	ErrorCode ResultCode::SetResultSCE(int sceError, bool log, const char* message)
	{
		m_LastErrorSCE = sceError;
		m_LastError = ProcessSceResult(sceError, log, message);
		return m_LastError;
	}

	ErrorCode ResultCode::SetResultSCE(int sceError, bool log)
	{
		m_LastErrorSCE = sceError;
		m_LastError = ProcessSceResult(sceError, log);
		return m_LastError;
	}

	ErrorCode ResultCode::SetResult(ErrorCode errorCode, int sceError, bool log, const char* function, int line, const char* message)
	{
		m_LastErrorSCE = sceError;
		m_LastError = ProcessResult(errorCode, log, m_className, function, line, message);
		return m_LastError;
	}

	ErrorCode ResultCode::SetResult(ErrorCode errorCode, bool log, const char* function, int line, const char* message)
	{
		m_LastErrorSCE = 0;
		m_LastError = ProcessResult(errorCode, log, m_className, function, line, message);
		return m_LastError;
	}

	ErrorCode ResultCode::SetResult(ErrorCode errorCode, bool log, const char* message)
	{
		m_LastErrorSCE = 0;
		m_LastError = ProcessResult(errorCode, log, message);
		return m_LastError;
	}

	ErrorCode ResultCode::SetResult(ErrorCode errorCode, bool log)
	{
		m_LastErrorSCE = 0;
		m_LastError = ProcessResult(errorCode, log);
		return m_LastError;
	}
}
